const formEl = document.querySelector(".form");
const inputEl = document.querySelector(".input");
const ulEl = document.querySelector(".list");
const clearBtn = document.querySelector(".clear-btn");

let list = JSON.parse(localStorage.getItem("list")) || [];

// Render tasks on page load
list.forEach((task) => renderTask(task));

formEl.addEventListener("submit", (event) => {
  event.preventDefault();
  if (inputEl.value.trim() === "") return;
  const task = { name: inputEl.value, checked: false };
  list.push(task);
  renderTask(task);
  updateLocalStorage();
  inputEl.value = "";
});

function renderTask(task) {
  const liEl = document.createElement("li");
  liEl.className = task.checked ? "checked" : "";
  liEl.innerHTML = `
    <span>${task.name}</span>
    <div>
      <i class="fas fa-check-square"></i>
      <i class="fas fa-trash"></i>
    </div>
  `;

  ulEl.appendChild(liEl);

  const checkBtn = liEl.querySelector(".fa-check-square");
  const trashBtn = liEl.querySelector(".fa-trash");

  checkBtn.addEventListener("click", () => {
    liEl.classList.toggle("checked");
    updateLocalStorage();
  });

  trashBtn.addEventListener("click", () => {
    liEl.remove();
    updateLocalStorage();
  });

  // Drag and Drop
  liEl.draggable = true;
  liEl.addEventListener("dragstart", (e) => {
    liEl.classList.add("dragging");
  });
  liEl.addEventListener("dragend", (e) => {
    liEl.classList.remove("dragging");
    updateLocalStorage();
  });
}

// Drag & Drop sorting
ulEl.addEventListener("dragover", (e) => {
  e.preventDefault();
  const dragging = document.querySelector(".dragging");
  const afterElement = getDragAfterElement(ulEl, e.clientY);
  if (afterElement == null) {
    ulEl.appendChild(dragging);
  } else {
    ulEl.insertBefore(dragging, afterElement);
  }
});

function getDragAfterElement(container, y) {
  const draggableElements = [
    ...container.querySelectorAll("li:not(.dragging)"),
  ];
  return draggableElements.reduce(
    (closest, child) => {
      const box = child.getBoundingClientRect();
      const offset = y - box.top - box.height / 2;
      if (offset < 0 && offset > closest.offset) {
        return { offset: offset, element: child };
      } else {
        return closest;
      }
    },
    { offset: Number.NEGATIVE_INFINITY }
  ).element;
}

// Clear all tasks
clearBtn.addEventListener("click", () => {
  ulEl.innerHTML = "";
  list = [];
  updateLocalStorage();
});

function updateLocalStorage() {
  const tasks = [];
  document.querySelectorAll(".list li").forEach((li) => {
    tasks.push({
      name: li.querySelector("span").innerText,
      checked: li.classList.contains("checked"),
    });
  });
  localStorage.setItem("list", JSON.stringify(tasks));
}
